/** @file   door.cpp
 * @brief   Implementation of Door - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "door.h"
#include "soundsamples.h"
#include "warglobals.h"
#include "tileid.h"
using namespace eng2d;
using std::string;
using std::vector;


namespace WeWantWar {

//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

vector<int> Door::SMALL_VUPPER_DOOR_BLOCKS;
vector<int> Door::SMALL_VLOWER_DOOR_BLOCKS;
vector<int> Door::SMALL_HLEFT_DOOR_BLOCKS;
vector<int> Door::SMALL_HRIGHT_DOOR_BLOCKS;

const Door::Orientation Door::VERTICAL;
const Door::Orientation Door::HORIZONTAL;

const Door::Key Door::NO_KEY;
const Door::Key Door::YELLOW_KEY;
const Door::Key Door::BLUE_KEY;

const int Door::STAYOPEN_DELAY;



/** Inits the door animations.
 */
int Door::initDoorBlocks( std::istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "</doors>" ) {
      return 0;
      
    } else if ( tmp == "<small_v_door>" ) {
      if ( initSmallVDoor( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<small_h_door>" ) {
      if ( initSmallHDoor( rIn ) != 0 ) {
        return -1;
      }
      
    }
  }
}



/** Reads the information about the small vertical door
 */
int Door::initSmallVDoor( std::istream& rIn )
{
  SMALL_VUPPER_DOOR_BLOCKS.clear();
  SMALL_VLOWER_DOOR_BLOCKS.clear();

  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "<upper_half>" ) {
      if ( storeFrameTable( rIn, SMALL_VUPPER_DOOR_BLOCKS, "</upper_half>" ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<lower_half>" ) {
      if ( storeFrameTable( rIn, SMALL_VLOWER_DOOR_BLOCKS, "</lower_half>") != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "</small_v_door>" ) {
      break;
    }
  }
  
  return 0;
}



/** Reads the Small Horizontal Door data
 */
int Door::initSmallHDoor( std::istream& rIn )
{
  SMALL_HLEFT_DOOR_BLOCKS.clear();
  SMALL_HRIGHT_DOOR_BLOCKS.clear();

  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }

    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );

    } else if ( tmp == "<left_half>" ) {
      if ( storeFrameTable( rIn, SMALL_HLEFT_DOOR_BLOCKS, "</left_half>") != 0 ) {
        return -1;
      }

    } else if ( tmp == "<right_half>" ) {
      if ( storeFrameTable( rIn, SMALL_HRIGHT_DOOR_BLOCKS, "</right_half>") != 0 ) {
        return -1;
      }

    } else if ( tmp == "</small_h_door>" ) {
      break;
    }
  }

  return 0;
}



/** Tells if the block on given block coordinates is a door.
 */
bool Door::isDoorBlock( int x, int y )
{
  Map::Block* pB = Map::blockAt( x, y, Map::IN_BLOCKS );
  if ( pB->user5 == TileID::id[TileID::ID_VS_DOOR] ||
       pB->user5 == TileID::id[TileID::ID_HS_DOOR] ) {
    return true;
  }
  return false;
}



/** Tells if the given block is a door
 */
bool Door::isDoorBlock( Map::Block* pB )
{
  if ( pB->user5 == TileID::id[ TileID::ID_VS_DOOR ] ||
       pB->user5 == TileID::id[ TileID::ID_HS_DOOR ] ) {
    return true;
  }
  return false;
}



/** Returns the orientation of the door at given block coordinate
 */
Door::Orientation Door::getOrientation( int x, int y )
{
  Map::Block* pB = Map::blockAt(x,y, Map::IN_BLOCKS);
  if ( pB->user5 == TileID::id[ TileID::ID_VS_DOOR ] ) {
    return VERTICAL;
  }
  return HORIZONTAL;
}



//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Constructs new door.
 */
Door::Door( Type t, int x, int y ) :
  m_type( t ),
  m_posX( x ),
  m_posY( y ),
  m_counter( DOOR_FRAME_DELAY ),
  m_frame( 0 ),
  m_state( STATE_CLOSED )
{
  m_orientation = getOrientation(x,y);
}


/** Destructor
 */
Door::~Door()
{
}




//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

/** Updates the door. The only thing this method does is to show
 * the opening door animation.
 */
void Door::update()
{
  switch ( m_type ) {
    case ( SMALL ): {
      this->updateSmallDoor();
      break;
    }
  }
}




/** Tells this door to open
 */
void Door::open()
{
  if ( m_state == STATE_OPEN ) {
    m_counter = STAYOPEN_DELAY;
    return;
  }
  if ( m_state == STATE_CLOSED || m_state == STATE_CLOSING ) {
    Sound::playSample( SMP_DOOR1, false );
    
  }
  m_state = STATE_OPENING;
}



/** Closes the door
 */
void Door::close()
{
  if ( m_state == STATE_CLOSED ) {
    return;
  }
  
  // If there is someone in the doorway we will not close the door.
  if ( WarGlobals::pObjTable->objectsInsideRect( this->getDoorArea() ) == true ) {
    return;
  }
  Sound::playSample( SMP_DOOR1, false );
  m_state = STATE_CLOSING;
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************


/** Returns the middlepoint x of this door in pixels
 */
float Door::getMidX() const
{
  int bs = Map::getBlockWidth();
  float mx;
  
  if ( m_orientation == VERTICAL ) {
    mx = (m_posX * bs) + (bs / 2) ;
  } else {
    mx = (m_posX * bs) + bs;
  }
  return mx;
}



/** Gets the middlepoint y of this door in pixels
 */
float Door::getMidY() const
{
  int bs = Map::getBlockHeight();
  float my;
  
  if ( m_orientation == VERTICAL ) {
    my = (m_posY * bs) + bs;
  } else {
    my = (m_posY * bs) + (bs / 2);
  }
  return my;
}



/** Returns the door area
 */
Rect2D Door::getDoorArea() const
{
  Rect2D r;
  if ( m_orientation == VERTICAL ) {
    Vec2D tl( m_posX*32, m_posY*32 );
    Vec2D br( 32, 64 );
    br += tl;
    r.topleft = tl;
    r.bottomright = br;
    
  } else {
    Vec2D tl( m_posX*32, m_posY*32 );
    Vec2D br( 64, 32 );
    br += tl;
    r.topleft = tl;
    r.bottomright = br;
    
  }
  return r;
}



/** Returns the state
 */
Door::State Door::state() const
{
  return m_state;
}




//********************************************************************
//                                                                   *
//      Private methods                                              *
//                                                                   *
//********************************************************************

/** Changes the door frame
 */
void Door::updateSmallDoor()
{
  m_counter -= 1;
  if ( m_counter >= 0 ) {
    return;
  }
  m_counter = DOOR_FRAME_DELAY;
  
  switch ( m_state ) {
    case ( STATE_OPEN ): {
      // We are open and we've waited the stayopen delay. Now we close
      // the door.
      this->close();
      break;
    }
    case ( STATE_OPENING ): {
      // We're opening.
      this->animateSmallDoor();
      break;
    }
    case ( STATE_CLOSING ): {
      // We're closing. If there are anyone in the doorway, we reopen
      // us.
      if ( WarGlobals::pObjTable->objectsInsideRect( this->getDoorArea() ) == true ) {
        this->open();
      }
      this->animateSmallDoor();
    }
  }
}



/** Animates the small opening door.
 */
void Door::animateSmallDoor()
{
  if ( m_state == STATE_OPENING ) {
    m_frame += 1;
  } else if ( m_state == STATE_CLOSING ) {
    m_frame -= 1;
  } else {
    return;
  }
  
  
  vector<int>* pBlocks1 = 0;
  vector<int>* pBlocks2 = 0;
  int offsetX = 0;
  int offsetY = 0;
  
  if ( m_orientation == VERTICAL ) {
    pBlocks1 = &SMALL_VUPPER_DOOR_BLOCKS;
    pBlocks2 = &SMALL_VLOWER_DOOR_BLOCKS;
    offsetX = 0;
    offsetY = 1;
  } else {
    pBlocks1 = &SMALL_HLEFT_DOOR_BLOCKS;
    pBlocks2 = &SMALL_HRIGHT_DOOR_BLOCKS;
    offsetX = 1;
    offsetY = 0;
  }
  
  // Have we shown the opening animation.
  if ( m_frame < 0 ) {
    // Frame index is negative so the door was closing and is now
    // closed.
    m_state = STATE_CLOSED;
    m_counter = 1;
    m_frame = 0;
    return;
    
  } else if ( m_frame >= pBlocks1->size() ) {
    // Door was opening and is now open.
    m_state = STATE_OPEN;
    m_counter = STAYOPEN_DELAY;
    m_frame = pBlocks1->size() - 1;
    return;
  }
  
  // Modify the map blocks to generate the door opening animation.
  int newBlock1 = pBlocks1->at( m_frame );
  int newBlock2 = pBlocks2->at( m_frame );
  Map::setBlock( m_posX, m_posY, Map::IN_BLOCKS, newBlock1 );
  Map::setBlock( m_posX + offsetX, m_posY + offsetY, Map::IN_BLOCKS, newBlock2 );
}



/** Reads the animation tiles of the opening doors
 */
int Door::storeFrameTable( std::istream& rIn, vector<int>& rVec,
                           const string endTag )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == endTag ) {
      return 0;
      
    } else {
      rVec.push_back( atoi( tmp.c_str() ) );
      
    }
  }
}

} // end of namespace
